/************************************************************************
 * ifchk.c                                                              *
 *                                                                      *
 * Copyright (C) 2002 - 2004 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/*****************************************************************
 * Project ifchk -- Host based network interface promiscuous     *
 *                  mode detection and handling.                 *
 *                                                               *
 * Version:         0.95b2                                       *
 *                                                               *
 * ------------------------------------------------------------- *
 *                                                               *
 * ifchk.c          ifchk program implementation.                *
 *                                                               *
 * Module                                                        *
 * Functions:     - Processes command line arguments.            *
 *                - Gets, sets and displays per-interface status *
 *                  flags.                                       *
 *                - Gets and displays per-interface traffic      *
 *                  metrics.                                     *
 *                - Logs interface status, metric dump display   *
 *                  and privileged operation (ie. interface      *
 *                  shutdown) violations.                        *
 *                - Performs invoking UID identification.        *
 *                - Displays a program usage message.            *
 *                - Displays ifchk revision and runtime data.    *
 *****************************************************************/
 
/**********************
 * include directives *
 **********************/

#include "ifchk.h"


/*******************************************************
 * main()      - Performs invoking UID identification, *
 *               command line argument sanity checking *
 *               and processing.                       *
 *                                                     *
 * Parameters:     argc: command line argument count.  *
 *               **argv: command line argument vector. *
 *                                                     *
 * Returns:      0 on success.                         *
 *               -1 on failure.                        *
 *******************************************************/

int main( int argc, char **argv )
{
    int opt = 0;		/* For getopt() processing. */
    char *argPtr = *argv;	/* For argument sanity check. */

    /*
     * First, get invoking UID info - required for subsequent operations.
     */
    if( ( pw = credentials() ) == NULL )
    {
        fprintf( stderr, "ifchk: ERROR: could not get passwd entry structure\n" );
        exit( EXIT_FAILURE );
     }

    if( argc == 1 )
    {
        nullFlag = TRUE;

	if( ifStat() != 0 )
        {
            exit( EXIT_FAILURE );
         }
     }

    if( argc == 2 )
    {
        /*
         * Argument sanity check.
         */
        while( *argPtr != '\000' )
        {
            argPtr++;
         }
            if( ( ( *( argPtr + 1 ) == '\055' ) &&
                  ( *( argPtr + 2 ) == '\000' ) ) ||
                ( ( *( argPtr + 1 ) == '\055' ) &&
                  ( *( argPtr + 2 ) == '\055' ) ) ||
                ( ( *( argPtr + 1 ) != '\055' ) ) &&
                  ( *( argPtr + 1 ) != '\000' ) ||
                ( ( *( argPtr + 1 ) == '\055' ) &&
                  ( *( argPtr + 2 ) != '\000' ) &&
                  ( *( argPtr + 3 ) != '\000' ) ) )
            {
                fprintf( stderr, "ifchk: ERROR: malformed argument list\n" );

                if( ifchkHelp() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
                    exit( EXIT_FAILURE );
                 }

                exit( EXIT_FAILURE );
             }
     }

    if( argc > 2 )
    {
        fprintf( stderr, "ifchk: ERROR: malformed argument list\n" );

        if( ifchkHelp() != 0 )
        {
            fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
            exit( EXIT_FAILURE );
         }

        exit( EXIT_FAILURE );
     }

    /*
     * Process command line arguments.
     */
    while( ( opt = getopt( argc, argv, "dhiv" ) ) != -1 )
    {
        switch( opt )
        {
            case 'd':
                dFlag = TRUE;
                {
                    if( ( pw -> pw_uid ) != 0 )
                    {
                        fprintf( stderr, "ifchk: ERROR: interface shutdown requires UID 0\n" );

                        if( ( writeLog( LOGWARNING, NULL, NULLSTATE ) ) != 0 )
                        {
                            fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                            exit( EXIT_FAILURE );
                         }

                        exit( EXIT_FAILURE );
                     }
                        
                    else
                    {
                        if( ifStat() != 0 )
                        {
                            exit( EXIT_FAILURE );
                         }
                     }
                 }
                break;

            case 'i':							
                iFlag = TRUE;
                if( ifStat() != 0 )
                {
                    exit( EXIT_FAILURE );
                 }
                break;

            case 'h':
                if( ifchkHelp() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
                    exit( EXIT_FAILURE );
                 }
                break;

            case 'v':
                if( ifchkRev() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program version\n" );
                    exit( EXIT_FAILURE );
                 }
                break;

            case '?':
                if( ifchkHelp() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
                    exit( EXIT_FAILURE );
                 }

                exit( EXIT_FAILURE );
         }
     }

    return (0);
}


/*******************************************************
 * ifStat()    - Performs interface status discovery,  *
 *               disabling & ifMetric() preprocessing. *
 *               Some code adapted from Unix Network   *
 *               Programming, Volume 1 by R. Stevens.  *
 *                                                     *
 * Parameters:   None.                                 *
 *                                                     *
 * Returns:      0 on success.                         *
 *               -1 on error.                          *
 *******************************************************/

int ifStat()
{
    int sockfd = 0;				/* Socket descriptor for ioctl(). */
    int ifCount = 0;				/* Interface count. */
    int i = 0;					/* For ifMetric(). */
    int j = 0;					/* Interface flags mask. */
    char *kBuf = '\000';			/* Points to configured interface list. */
    char *kBufPtr = '\000';			/* For ifreq struct traversal. */
    struct ifconf ifc;				/* Configured interface list. */
    struct ifreq *ifr;				/* Per-interface characteristics. */
    struct ifList *head = NULL;			/* Linked list of interface */
    struct ifList *tail = NULL;			/* name/unit number pairings. */
    struct ifList *new = NULL;			/* Constructed locally & passed */
    struct ifList *cur = NULL;			/* to ifMetric() for processing */
    struct ifList *freeList = NULL;		/* & output. */

    /*
     * Socket for ioctl() calls.
     */
    if( ( sockfd = socket( AF_INET, SOCK_DGRAM, 0 ) ) == -1 )
    {
        perror( "socket" );
        return (-1);
     }

    /*
     * Allocate memory & get configured interface list.
     */
    if( ( kBuf = malloc( KBUFSZ ) ) == NULL )
    {
        perror( "malloc" );
        return (-1); 
     }

    ifc.ifc_len = KBUFSZ;
    ifc.ifc_buf = kBuf;

    if ( ioctl( sockfd, SIOCGIFCONF, &ifc ) < 0 )
    {
        perror( "ioctl" );
        return (-1); 
     }

    /*
     * Interface count. Failure here means KBUFSZ (& by extention
     * ifc.ifc_len) is too small to hold configured interface list.
     */
    if( ( ifCount = ifc.ifc_len / sizeof( struct ifreq ) ) == 0 )
    {
        fprintf( stderr, "ifchk: ERROR: constant KBUFSZ (ifchk.h:77:) too small\n" );
        return (-1);
     }

    if( iFlag != TRUE )
    {
        printf( "interface(s): %d\n", ifCount );

        if( ( writeLog( LOGINFO, NULL, STATSTATE ) ) != 0 )
        {
            fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
            return (-1);
         }
     }

    /*
     * Loop through all interfaces. Either we're examining per-interface
     * flags or gathering interface names to pass to ifMetric().
     */
    for( kBufPtr = kBuf; kBufPtr < kBuf + ifc.ifc_len; kBufPtr += 32 )
    {
        ifr = ( struct ifreq * ) kBufPtr;

        if( ioctl( sockfd, SIOCGIFFLAGS, ( char * ) ifr ) < 0 )
        {
            perror( "ioctl" );
            return (-1);
         }

        j = ifr -> ifr_flags;

        /*
         * Interface is promiscuous - print & log state.
         */
        if( ( ( j & IFF_PROMISC ) == 0x100 ) && ( nullFlag == TRUE ) )
        {
            printf( "%4s: PROMISC\n", ifr -> ifr_name );

            if( ( writeLog( LOGWARNING, ifr -> ifr_name, PROMISC ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        /*
         * Interface is promiscuous - disable it, print & log state.
         */
        else if( ( ( j & IFF_PROMISC ) == 0x100 ) && ( dFlag == TRUE ) )
        {
            printf( "%4s: PROMISC [*]\n", ifr -> ifr_name );

            ifr -> ifr_ifru.ifru_flags &= ~(IFF_UP|IFF_RUNNING);

            if( ioctl( sockfd, SIOCSIFFLAGS, ( char * ) ifr ) < 0 )
            {
                perror( "ioctl" );
                return (-1);
             }

            if( ( writeLog( LOGWARNING, ifr -> ifr_name, PROMISC_DISABLE ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        /*
         * Construct linked list of interface names for ifMetric().
         */
        else if( iFlag == TRUE )
        {
            if( ( new = malloc( sizeof( struct ifList ) ) ) == NULL )
            {
                perror( "malloc" );
                return (-1);
             }

            /*
             * Create a new node.
             */
            memset( new, '\000', sizeof( new ) );
            strncpy( new -> data, ifr -> ifr_name, sizeof( new -> data ) );
            new -> data[DATASZ - 1] = '\000';
            cur = new;

            /*
             * Append new node to list end.
             */
            if( head == NULL )
            {
                head = cur; 
             }

            else
            {
                tail -> next = cur;
             }

            tail = cur;
            cur -> next = NULL;

            i++;

            /*
             * When the list is complete, call ifMetric()
             * passing it a pointer to the head node.
             */
            if( i == ifCount )
            {
                if( ifMetric( head ) != 0 )
                {
                    return (-1);
                 }

                /*
                 * Deallocate list memory after use.
                 */
                for( cur = head; cur != NULL; cur = freeList )
                {
                    freeList = cur -> next;
                    free( cur );
                 }

                free( new );
             }
         }

        /*
         * Interface is up - print & log state.
         */
        else if( ( j & IFF_UP ) == 0x1 )
        {
            printf( "%4s: normal\n", ifr -> ifr_name );
            
            if( ( writeLog( LOGINFO, ifr -> ifr_name, NORMAL ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        /*
         * Interface is down - print & log state.
         */
        else
        {
            printf( "%4s: *down*\n", ifr -> ifr_name );
            
            if( ( writeLog( LOGINFO, ifr -> ifr_name, DOWN ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }   
     }

    /*
     * Clean-up via close() & free().
     */
    if( close( sockfd ) != 0 )
    {
        perror( "close" );
        return (-1);
     }

    free( kBuf );

    return (0);
}


/********************************************************
 * ifMetric()  - Generates and prints interface traffic *
 *               metrics for all configured interfaces. *
 *               Some code adapted from Unix Network    *
 *               Programming, Volume 1 by R. Stevens.   * 
 *                                                      *
 * Parameters:   *list: first node of linked list of    *
 *               interface name/unit number pairings.   *
 *                                                      *
 * Returns:      0 on success.                          *
 *               -1 on failure.                         *
 ********************************************************/

int ifMetric( struct ifList *list ) 
{
    size_t bufSize;				/* Amount of data kernel returns via sysctl(). */
    int mib[6];					/* sysctl() request to kernel. */
    char *buf;					/* What kernel returns after sysctl(). */
    char *lim;					/* Points just beyond last byte of data returned. */
    char *nextMsg;				/* For ifm struct traversal. */
    struct if_msghdr *ifm;			/* Interface metrics. */
    struct ifList *cur;				/* Current node in linked list. */

    mib[0] = CTL_NET;				/* Kernel subsystem to direct sysctl() to. */
    mib[1] = AF_ROUTE;				/* Which part of CTL_NET to look at. */
    mib[2] = 0;					/* Protocol number. */
    mib[3] = 0;					/* Address family. */
    mib[4] = NET_RT_IFLIST;			/* Return info on all configured interfaces. */
    mib[5] = 0;					/* If != 0, return info on _an_ interface. */

    /*
     * Determine amount of kernel data returned via sysctl().
     */
    if( ( sysctl( mib, 6, NULL, &bufSize, NULL, 0 ) < 0 ) )
    {
        perror( "sysctl" );
        return (-1);
     }

    /*
     * Create suitably sized buffer based on the above.
     */
    if( ( buf = malloc( bufSize ) ) == NULL )
    {
        perror( "malloc" );
        return (-1);
     }

    /*
     * Call sysctl() again to get metrics data.
     */
    if( ( sysctl( mib, 6, buf, &bufSize, NULL, 0 ) < 0 ) )
    {
        perror( "sysctl" );
        return (-1);
     }

    /*
     * Prepare to display metrics data.
     */
    lim = buf + bufSize;

    cur = list;

    if( cur -> data == NULL )
    {
        fprintf( stderr, "ifchk: ERROR: interface name/unit number list is empty\n" );
        return (-1);
     }

    printf( "***   Network Interface Metrics   ***\n" );
    printf( "Name	Index	Ipkts		Opkts\n" );

    nextMsg = buf;

    /*
     * Loop through all messages & output 1 line of
     * metrics data for each interface.
     */
    while( nextMsg < lim )
    {
        ifm = ( struct if_msghdr * ) nextMsg;

        if( ifm -> ifm_type == RTM_IFINFO ) 
        {
           fprintf( stdout, "%s	%-7d %-13u   %-13u\n", cur -> data, ifm -> ifm_index,
                    ifm -> ifm_data.ifi_ipackets, ifm -> ifm_data.ifi_opackets );

           if( cur -> next != NULL )
           {
               cur = cur -> next;
            }
         }

        free( buf );

        nextMsg += ifm -> ifm_msglen;
     }

    /*
     * Log that a metrics dump was performed.
     */
    if( ( writeLog( LOGINFO, NULL, NULLSTATE ) ) != 0 )
    {
        fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
        return (-1);
     }

    return (0);
}


/*****************************************************
 * writeLog()  - Performs logging of ifchk activity. *
 *                                                   *
 * Parameters:   priority: priority at which to log. *
 *               *object:  an interface name.        *
 *               ifState:  an interface state.       *
 *                                                   *
 * Returns:      0 on success.                       *
 *               -1 on failure.                      *
 *****************************************************/

int writeLog( int priority, char *object, int ifState )
{
    openlog( "ifchk", LOG_PID, LOG_USER );

    /*
     * Interface is down.
     */
    if( ( priority == LOGINFO ) && ( ifState == DOWN ) )
    {
        syslog( LOG_INFO, "%4s: *down*", object );
     }

    /*
     * Interface is normal/up.
     */
    else if( ( priority == LOGINFO ) && ( ifState == NORMAL ) )
    {
        syslog( LOG_INFO, "%4s: normal", object );
     }

    /*
     * Interface is promiscuous.
     */
    else if( ( priority == LOGWARNING ) && ( ifState == PROMISC ) )
    {
        syslog( LOG_WARNING, "%4s: PROMISC", object );
     }
    
    /*
     * Interface is promiscuous, disabled.
     */
    else if( ( priority == LOGWARNING ) && ( ifState == PROMISC_DISABLE ) )
    {
        syslog( LOG_WARNING, "%4s: PROMISC [*]", object );
     }
    
    /*
     * Interface status dump performed.
     */
    else if( ( priority == LOGINFO ) && ( ifState == STATSTATE ) )
    {
        syslog( LOG_INFO, "interface status: dump to stdout initiated by user %s (UID: %d)",
                                                                 pw -> pw_name, pw -> pw_uid );
     }

    /*
     * Metrics dump performed.
     */
    else if( ( priority == LOGINFO ) && ( ifState == NULLSTATE ) )
    {
        syslog( LOG_INFO, "interface metrics: dump to stdout initiated by user %s (UID: %d)",
                                                                  pw -> pw_name, pw -> pw_uid );
     }

    /*
     * Non privileged interface shutdown attempt.
     */
    else if( ( priority == LOGWARNING ) && ( ifState == NULLSTATE ) )
    {
        syslog( LOG_WARNING, "interface shutdown: attempt by non privileged user %s (UID: %d)",
                                                                    pw -> pw_name, pw -> pw_uid );
     }

    /*
     * Default: signals an error condition.
     */
    else
    {
        syslog( LOG_WARNING, "%4s: interface state logging failure", object );
        return (-1);
     }

    closelog();

    return (0);
}


/*****************************************************
 * credentials() - Returns credentials data on the   *
 *                 user invoking ifchk.              *
 *                                                   *
 * Parameters:     None.                             *
 *                                                   *
 * Returns:        struct of type passwd on success. *
 *                 NULL on failure.                  *
 *****************************************************/

struct passwd *credentials()
{
    struct passwd *pwData = NULL;

    /*
     * Do we have a UID 0 password database entry to authenticate 
     * with for privileged operations (ie. interface shutdown)?
     */
    if( ( pwData = getpwuid( 0 ) ) == NULL)
    {
        perror( "getpwuid" );
        return (NULL);
     }

    if( pwData -> pw_uid != 0 ) 
    {
        fprintf( stderr, "ifchk: ERROR: UID 0 does not exist in password database\n" );
        return (NULL);
     }

    /*
     * If we do, check that the UID 0 name field entry is "root".
     */
    if( ( strncmp( pwData -> pw_name, "root", sizeof( "root" ) ) ) != 0 )
    {
        printf( "ifchk: WARNING: UID 0 name field entry in password database is \"%s\"\n",
                                                                         pwData -> pw_name );
     }

    /*
     * With UID 0 check done, clean house and get UID that ran ifchk.
     */
    pwData = NULL;

    if( ( pwData = getpwuid( getuid() ) ) == NULL )
    {
        perror( "getpwuid" );
        return (NULL);
     }

    return (pwData);
}


/*****************************************
 * ifchkHelp() - Prints program options. *
 *                                       *
 * Parameters:   None.                   *
 *                                       *
 * Returns:      0 on success.           *
 *****************************************/

int ifchkHelp()
{
    printf( "Usage: ifchk [option]\n" );
    printf( "Options:\n" );
    printf( "  -d   shutdown all interfaces running in\n" );
    printf( "       promiscuous mode (operation requires\n" );
    printf( "       super-user privileges)\n" );
    printf( "  -i   display network traffic metrics for\n" );
    printf( "       all configured interfaces\n" );
    printf( "  -h   print this usage message\n" );
    printf( "  -v   print program revision information,\n" );
    printf( "       host runtime environment data and\n" );
    printf( "       program build date/time stamp\n" );

    return (0);
}


/****************************************************
 * ifchkRev()  - Prints program revision, execution *
 *               environment info & build date/time *
 *               stamp.                             *
 *                                                  *
 * Parameters:   None.                              *
 *                                                  *
 * Returns:      0 on success.                      *
 *               -1 on failure.                     *
 ****************************************************/

int ifchkRev()
{
    struct utsname nodeId;				/* uname structure. */
    const char buildDate[] = ""__DATE__" "__TIME__"";	/* ifchk build date and time. */

    /*
     * Get information identifying the current system.
     */
    if( ( uname( &nodeId ) ) == -1 )
    {
        perror( "uname" );
        return (-1);
     }

    /*
     * Print the above along with ifchk build date/time.
     */
    printf( "ifchk %s on %s, %s %s %s\n", VERS, nodeId.nodename, nodeId.sysname,
                                                  nodeId.release, nodeId.machine );
    printf( "build date: %s\n", buildDate );

    return (0);
}
 
