/************************************************************************
 * ifchk.c                                                              *
 *                                                                      *
 * Copyright (C) 2002 - 2004 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/**************************************************************
 * Project ifchk -- Host based network interface promiscuous  *
 *                  mode detection and handling.              *
 *                                                            *
 * Version:         0.95b2                                    *
 *                                                            *
 * ---------------------------------------------------------- *
 *                                                            *
 * ifchk.c          ifchk program framework/front-end.        *
 *                                                            *
 * Module                                                     *
 * Functions:     - Processes command line arguments.         *
 *                - Provides a point of entry into the ifchk  *
 *                  program core.                             *
 *                - Displays a program usage message.         *
 *                - Displays ifchk revision and runtime data. *
 **************************************************************/
 
/**********************
 * include directives *
 **********************/

#include "ifchk.h"


/***********************
 * global declarations *
 ***********************/

struct passwd *pw = NULL;	/* Structure for user credential check. */
int dFlag = FALSE;		/* Interface disable flag. */
int iFlag = FALSE;		/* Interface metrics flag. */
int nullFlag = FALSE;		/* Dummy interface flag - for processFlags(). */


/*******************************************************
 * main()      - Performs invoking UID identification, * 
 *               command line argument sanity checking *
 *               and processing.                       *
 *                                                     *
 * Parameters:     argc: command line argument count.  *
 *               **argv: command line argument vector. *
 *                                                     *
 * Returns:      0 on success.                         *
 *               -1 on failure.                        *
 *******************************************************/

int main( int argc, char **argv )
{
    int opt = 0;		/* For getopt() processing. */
    char *argPtr = *argv;	/* For argument sanity check. */

    /*
     * First, get invoking UID info - required for subsequent operations.
     */
    if( ( pw = credentials() ) == NULL )
    {
        fprintf( stderr, "ifchk: ERROR: could not get passwd entry structure\n" );
        exit( EXIT_FAILURE );
     }

    if( argc == 1 )
    {
        nullFlag = TRUE;

	if( ifStat() != 0 )
        {
            exit( EXIT_FAILURE );
         }
     }

    if( argc == 2 )
    {
        /*
         * Argument sanity check.
         */
        while( *argPtr != '\000' )
        {
            argPtr++;
         }
            if( ( ( *( argPtr + 1 ) == '\055' ) &&
                  ( *( argPtr + 2 ) == '\000' ) ) ||
                ( ( *( argPtr + 1 ) == '\055' ) &&
                  ( *( argPtr + 2 ) == '\055' ) ) ||
                ( ( *( argPtr + 1 ) != '\055' ) ) &&
                  ( *( argPtr + 1 ) != '\000' ) ||
                ( ( *( argPtr + 1 ) == '\055' ) &&
                  ( *( argPtr + 2 ) != '\000' ) &&
                  ( *( argPtr + 3 ) != '\000' ) ) )
            {
                fprintf( stderr, "ifchk: ERROR: malformed argument list\n" );

                if( ifchkHelp() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
                    exit( EXIT_FAILURE );
                 }

                exit( EXIT_FAILURE );
             }
     }

    if( argc > 2 )
    {
        fprintf( stderr, "ifchk: ERROR: malformed argument list\n" );

        if( ifchkHelp() != 0 )
        {
            fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
            exit( EXIT_FAILURE );
         }

        exit( EXIT_FAILURE );
     }

    /*
     * Process command line arguments.
     */
    while( ( opt = getopt( argc, argv, "dhiv" ) ) != -1 )
    {
        switch( opt )
        {
            case 'd':
                dFlag = TRUE;
                {
                    if( ( pw -> pw_uid ) != 0 )
                    {
                        fprintf( stderr, "ifchk: ERROR: interface shutdown requires UID 0\n" );

                        if( ( writeLog( LOGWARNING, NULL, NULLSTATE ) ) != 0 )
                        {
                            fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                            exit( EXIT_FAILURE );
                         }

                        exit( EXIT_FAILURE );
                     }
                        
                    else
                    {
                        if( ifStat() != 0 )
                        {
                            exit( EXIT_FAILURE );
                         }
                     }
                 }
                break;

            case 'i':
                iFlag = TRUE;
                if( ifStat() != 0 )
                {
                    exit( EXIT_FAILURE );
                 }
                break;

            case 'h':
                if( ifchkHelp() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
                    exit( EXIT_FAILURE );
                 }
                break;

            case 'v':
                if( ifchkRev() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program version\n" );
                    exit( EXIT_FAILURE );
                 }
                break;

            case '?':
                if( ifchkHelp() != 0 )
                {
                    fprintf( stderr, "ifchk: ERROR: could not print program options\n" );
                    exit( EXIT_FAILURE );
                 }

                exit( EXIT_FAILURE );
         }
     }

    return (0);
}


/******************************************************
 * ifStat()    - Entry point into an operating system *
 *               specific module that implements core *
 *               interface management functionality.  *
 *                                                    *
 * Parameters:   None.                                *
 *                                                    *
 * Returns:      0 on success.                        *
 *               -1 on failure.                       *
 ******************************************************/

int ifStat()
{
    extern struct nlSockStruct nlSock;	/* netlink socket structure. */

    /*
     * Enter the ifchk Linux module.
     */
    if( ( nlCreateSock( &nlSock ) ) != 0 )
    {
        return (-1);
     }

    if( ( nlSndMsg( &nlSock ) ) != 0 )
    {
        return (-1);
     }

    if( ( nlRcvMsg( &nlSock ) ) != 0 )
    {
        return (-1);
     }

    return (0);
}


/*****************************************
 * ifchkHelp() - Prints program options. *
 *                                       *
 * Parameters:   None.                   *
 *                                       *
 * Returns:      0 on success.           *
 *****************************************/

int ifchkHelp()
{
    printf( "Usage: ifchk [option]\n" );
    printf( "Options:\n" );
    printf( "  -d   shutdown all interfaces running in\n" );
    printf( "       promiscuous mode (operation requires\n" );
    printf( "       super-user privileges)\n" );
    printf( "  -i   display network traffic metrics for\n" );
    printf( "       all configured interfaces\n" );
    printf( "  -h   print this usage message\n" );
    printf( "  -v   print program revision information,\n" );
    printf( "       host runtime environment data and\n" );
    printf( "       program build date/time stamp\n" );

    return (0);
}


/****************************************************
 * ifchkRev()  - Prints program revision, execution *
 *               environment info & build date/time *
 *               stamp.                             *
 *                                                  *
 * Parameters:   None.                              *
 *                                                  *
 * Returns:      0 on success.                      *
 *               -1 on failure.                     *
 ****************************************************/
 

int ifchkRev()
{
    struct utsname nodeId;				/* uname structure. */
    const char buildDate[] = ""__DATE__" "__TIME__"";	/* ifchk build date and time. */

    /*
     * Get information identifying the current system.
     */
    if( ( uname( &nodeId ) ) == -1 )
    {
        perror( "uname" );
        return (-1);
     }

    /*
     * Print the above along with ifchk build date/time.
     */
    printf( "ifchk %s on %s, %s %s %s\n", VERS, nodeId.nodename, nodeId.sysname,
                                                  nodeId.release, nodeId.machine );
    printf( "build date: %s\n", buildDate );

    return (0);
}
