/************************************************************************
 * linux.c                                                              *
 *                                                                      *
 * Copyright (C) 2002 - 2004 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/**************************************************************
 * Project ifchk -- Host based network interface promiscuous  *
 *                  mode detection and handling.              *
 *                                                            *
 * Version:         0.95b2                                    *
 *                                                            *
 * ---------------------------------------------------------- *
 *                                                            *
 * linux.c:         ifchk program core for linux.             *
 *                  Some code adapted from the iproute2       *
 *                  package by Alexey Kuznetsov.              *
 *                                                            *
 * Module                                                     *
 * Functions:     - Performs netlink socket creation.         *
 *                - Gets and sets per-interface status flags. *
 *                - Displays per-interface status flags.      *
 *                - Gets per-interface traffic metrics.       *
 *                - Displays per-interface traffic metrics.   *
 **************************************************************/

/**********************
 * include directives *
 **********************/

#include "linux.h"


/**********************************************************
 * nlCreateSock() - Performs netlink socket creation and  *
 *                  initial setup for communication/data  *
 *                  transfer between ifchk and the        *
 *                  NETLINK_ROUTE linux kernel subsystem. *
 *                                                        *
 * Parameters:      *nlSock: struct of type nlSockStruct  *
 *                  who's members define netlink socket   *
 *                  attributes.                           *
 *                                                        *
 * Returns:         0 on success.                         *
 *                  -1 on failure.                        *
 **********************************************************/

int nlCreateSock( struct nlSockStruct *nlSock )
{
    int nameLen = 0;	/* getsockname() value-result argument. */

    memset( nlSock, '\000', sizeof( nlSock ) );

    /*
     * Socket for netlink related calls. 
     */
    if( ( nlSock -> nlsockfd = socket( AF_NETLINK, SOCK_DGRAM, NETLINK_ROUTE ) ) == -1 )
    {
        perror( "socket" );
        return (-1);
     }

    /*
     * Setup for getsockname() below.
     */
    memset( &nlSock -> client, '\000', sizeof( nlSock -> client ) );
    nlSock -> client.nl_family = AF_NETLINK;
    nlSock -> client.nl_pid = 0;
    nlSock -> client.nl_groups = 0;

    if( ( bind( nlSock -> nlsockfd, (struct sockaddr *)&nlSock -> client,
                                    sizeof( nlSock -> client ) )  ) == -1 )
    {
        perror( "bind" );
        return (-1);
     }

    nameLen = sizeof( nlSock -> client );

    /*
     * Get our PID to ensure that netlink messages
     * to be returned to us belong to our process.
     */
    if( ( getsockname( nlSock -> nlsockfd, (struct sockaddr *)&nlSock -> client, &nameLen ) ) == -1 )
    {
        perror( "getsockname" );
        return (-1);
     }

    if( nameLen != sizeof( nlSock -> client ) )
    {
        fprintf( stderr, "ifchk: ERROR: netlink socket address length mismatch\n" );
        return (-1);
     }

    /*
     * Additional PID-to-netlink message check.
     */
    nlSock -> sequence = time( NULL );

    return (0);
}


/************************************************************
 * nlSndMsg() - Creates and sends a netlink message to the  *
 *              kernel that initiates the return of network *
 *              interface specific attribute information.   *
 *                                                          *
 * Parameters:  *nlSock: struct of type nlSockStruct who's  *
 *              members define netlink socket attributes.   *
 *                                                          *
 * Returns:     0 on success.                               *
 *              -1 on failure.                              *
 ************************************************************/

int nlSndMsg( struct nlSockStruct *nlSock )
{
    struct nlMsgStruct nlMsg;	/* netlink message structure. */
    struct sockaddr_nl local;	/* netlink socket address structure. */ 

    memset( &local, '\000', sizeof( local ) );
    local.nl_family = AF_NETLINK;

    /*
     * Construct our netlink message.
     */
    nlMsg.hdr.nlmsg_len = sizeof( nlMsg );
    nlMsg.hdr.nlmsg_type = RTM_GETLINK;
    nlMsg.hdr.nlmsg_flags = NLM_F_ROOT | NLM_F_MATCH | NLM_F_REQUEST;
    nlMsg.hdr.nlmsg_pid = 0;
    nlMsg.hdr.nlmsg_seq = ++nlSock -> sequence;
    nlMsg.payload.rtgen_family = AF_PACKET;

    /*
     * Send our netlink message to the kernel.
     */
    if( ( sendto( nlSock -> nlsockfd, (void*)&nlMsg, sizeof( nlMsg ), 0,
                      (struct sockaddr*)&local, sizeof( local ) ) ) == 0 )
    {
        perror( "sendto" );
        return (-1);
     }

    return (0);
}


/***********************************************************
 * nlRcvMsg() - Manages the reception, from the kernel, of *
 *              netlink messages who's payloads contain    *
 *              interface specific attribute information.  *
 *              Creates a linked list of this information  *
 *              to hand off to processFlags(), ifMetric(). *
 *                                                         *
 * Parameters:  *nlSock: struct of type nlSockStruct who's *
 *              members define netlink socket attributes.  *
 *                                                         *
 * Returns:     0 on success.                              *
 *              -1 on failure.                             *
 ***********************************************************/

int nlRcvMsg( struct nlSockStruct *nlSock )
{
    char buf[IOVBUFSZ] = "";			/* Buffer for interface data. */
    struct sockaddr_nl local;			/* netlink socket address structure. */
    struct iovec iov = { buf, sizeof( buf ) };	/* Starting address and size of buf[]. */
    struct ifinfomsg *netif = NULL;		/* Interface attribute structure. */
    char netifName[IF_NAMESIZE] = "";		/* Interface name/unit number, e.g., "eth0". */
    int mBufLen = 0;				/* Bytes returned by recvmsg(). */
    struct nlmsghdr *msgHdr = NULL;		/* netlink message header. */
    struct nlmsghdr *ifCountMsgHdr = NULL;	/* netlink message header. */
    int ifCount = 0;				/* Interface count. */
    int ifCountBufLen = 0;			/* Bytes returned by recvmsg(). */ 
    int i = 0;					/* For calls to processFlags(), ifMetric(). */ 
    struct ifList *head = NULL;			/* Linked list of interface \ */
    struct ifList *tail = NULL;			/* name/unit number pairings \ */
    struct ifList *new = NULL;			/* and corresponding flags. */
    struct ifList *cur = NULL;
    struct ifList *freeList = NULL;
    struct msghdr msg =				/* For scatter/gather I/O. */
    {
        (void *)&local, sizeof( local ),
        &iov, 1, NULL, 0, 0
    };

    /*
     * Initial setup for computing interface count.
     */
    if( ( ifCountBufLen = recvmsg( nlSock -> nlsockfd, &msg, MSG_PEEK ) ) == -1 )
    {
        perror( "recvmsg" );
        return (-1);
     }

    if( ifCountBufLen == 0 )
    {
        fprintf( stderr, "ifchk: ERROR: EOF on netlink socket\n" );
        return (-1);
     }

    if( msg.msg_namelen != sizeof( local ) )
    {
        fprintf( stderr, "ifchk: ERROR: netlink socket address length mismatch\n" );
        return (-1);
     }

    /*
     * Failure here means buf[IOVBUFSZ] is too small to
     * hold configured interface data for above operation. 
     */
    if( msg.msg_flags & MSG_TRUNC )
    {
        fprintf( stderr, "ifchk: ERROR: constant IOVBUFSZ (linux.h:68:) too small\n" );
        return (-1);
     }

    ifCountMsgHdr = (struct nlmsghdr *)buf;

    /*
     * Compute and print interface count.
     */
    while( NLMSG_OK( ifCountMsgHdr, ifCountBufLen ) )
    {
        if( ifCountMsgHdr -> nlmsg_type != NLMSG_DONE )
        {
            ifCount++;
         }

        ifCountMsgHdr = NLMSG_NEXT( ifCountMsgHdr, ifCountBufLen );
     }

    /*
     * We didn't iterate through the whole netlink message stream.
     */
     if( ifCountBufLen )
     {
         fprintf( stderr, "ifchk: ERROR: netlink message remnant of %d bytes\n", ifCountBufLen );
         return (-1);
      }

    if( iFlag != TRUE )
    {
        printf( "interface(s): %d\n", ifCount );

        if( ( writeLog( LOGINFO, NULL, STATSTATE ) ) != 0 )
        {
            fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
            return (-1);
         }
     }

    /*
     * Master loop that drives interface list creation.
     */
    while( 1 )
    {
        /*
         * Initial setup for per-interface processing.
         */
        if( ( mBufLen = recvmsg( nlSock -> nlsockfd, &msg, 0 ) ) == -1 )
        {
            perror( "recvmsg" );
            return (-1);
         }

        if( mBufLen == 0 )
        {
            fprintf( stderr, "ifchk: ERROR: EOF on netlink socket\n" );
            return (-1);
         }

        if( msg.msg_namelen != sizeof( local ) )
        {
            fprintf( stderr, "ifchk: ERROR: netlink socket address length mismatch\n" );
            return (-1);
         }
        
        /*
         * Failure here means buf[IOVBUFSZ] is too small to
         * hold configured interface data for above operation.
         */
        if( msg.msg_flags & MSG_TRUNC )
        {
            fprintf( stderr, "ifchk: ERROR: constant IOVBUFSZ (linux.h:68:) too small\n" );
            return (-1);
         }

        msgHdr = (struct nlmsghdr *)buf;

        /*
         * Process the current interface.
         */
        while( NLMSG_OK( msgHdr, mBufLen ) )
        {
            int err = 0;

            /*
             * Is this netlink message really for us?
             */
            if( ( msgHdr -> nlmsg_pid != nlSock -> client.nl_pid ) ||
                        ( msgHdr -> nlmsg_seq != nlSock -> sequence ) )
            {
                fprintf( stderr, "ifchk: ERROR: received invalid message on netlink socket\n" );
                return (-1);
             }

            /*
             * We've reached the end of our netlink
             * message stream so clean up and exit.
             */
            if( msgHdr -> nlmsg_type == NLMSG_DONE )
            {
                if( ( close( nlSock -> nlsockfd ) ) != 0 )
                {
                    perror( "close" );
                    return (-1);
                 }

                return (0);
             }

            /*
             * Handle netlink specific error conditions.
             */
            if( msgHdr -> nlmsg_type == NLMSG_ERROR )
            {
                struct nlmsgerr *err = (struct nlmsgerr*)NLMSG_DATA( msgHdr );

                if( msgHdr -> nlmsg_len < NLMSG_LENGTH( sizeof( struct nlmsgerr ) ) )
                {
                    fprintf( stderr, "ifchk: ERROR: netlink message trucated\n" );
                 }

                else
                {
                    errno = -err -> error;
                    perror( "NLMSG_ERROR" );
                 }

                return (-1);
             }

            /*
             * Access the payload (contains interface data)
             * associated with the current netlink message.
             */ 
            netif = NLMSG_DATA( msgHdr );

            if( ( if_indextoname( netif -> ifi_index, netifName ) ) == NULL )
            {
                perror( "if_indextoname" );
                return (-1);
             }

            /*
             * Create a new list node for current interface.
             */
            if( ( new = malloc( sizeof( struct ifList ) ) ) == NULL )
            {
                perror( "malloc" );
                return (-1);
             }

            memset( new, '\000', sizeof( new ) );
            strncpy( new -> data, netifName, sizeof( new -> data ) );
            new -> flags = netif -> ifi_flags;
            new -> data[DATASZ - 1] = '\000';
            cur = new;

            /*
             * Append new node to list end.
             */
            if( head == NULL )
            {
                head = cur; 
             }

            else
            {
                tail -> next = cur;
             }

            tail = cur;
            cur -> next = NULL;

            i++;

            /*
             * Prepare to hand off to ifMetric() or
             * processFlags() upon list completion.
             */
            if( i == ifCount )
            {
                /*
                 * For dumping of per-interface metrics.
                 */
                if( iFlag == TRUE )
                {
                    if( ifMetric( head ) != 0 )
                    {
                        return (-1);
                     }
                 }

                else
                {
                    /*
                     * For displaying and/or setting interface flags.
                     */
                    if( processFlags( head, ifCount ) != 0 )
                    {
                        return (-1);
                     }
                 }

                /*
                 * Deallocate list memory after use.
                 */
                for( cur = head; cur != NULL; cur = freeList )
                {
                    freeList = cur -> next;
                    free( cur );
                 }
             }

            /*
             * Get the next netlink message.
             */
            msgHdr = NLMSG_NEXT( msgHdr, mBufLen );
         }

        /*
         * We didn't iterate through the whole netlink message stream.
         */
        if( mBufLen )
        {
            fprintf( stderr, "ifchk: ERROR: netlink message remnant of %d bytes\n", mBufLen );
            return (-1);
         }
     }

    return (0);
}


/**********************************************************
 * processFlags() - Performs interface status discovery,  * 
 *                  disabling and printing of interface   *
 *                  state.                                *
 *                                                        *
 * Parameters:      *ifList: first node of linked list of *
 *                  interface name/unit number pairings   *
 *                  and corresponding flags.              *
 *                                                        *
 *                  count: interface count.               *
 *                                                        *
 * Returns:         0 on success.                         *
 *                  -1 on failure.                        *
 **********************************************************/

int processFlags( struct ifList *list, int count )
{
    int i = 0;
    int ifCount = 0;			/* Count of all present interfaces. */
    int sockfd = 0;			/* Socket descriptor for ioctl(). */
    struct ifreq ifr;			/* Per-interface characteristics. */
    struct ifList *cur = NULL;		/* Current node in linked list. */

    ifCount = count;
    cur = list;

    /*
     * Exit if passed in interface list is empty.
     */
    if( ( cur -> data == NULL ) || ( ! cur -> flags ) )
    {
        fprintf( stderr, "ifchk: ERROR: interface list is incomplete\n" );
        return (-1);
     }

    /*
     * Loop through all interfaces and examine their flags.
     */
    while( i < ifCount ) 
    {
        /*
         * Interface is promiscuous - print & log state.
         */
        if( ( cur -> flags & IFF_PROMISC ) && ( nullFlag == TRUE ) )
        {
            printf( "%6s: PROMISC\n", cur -> data );

            if( ( writeLog( LOGWARNING, cur -> data, PROMISC ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        /*
         * Interface is promiscuous - disable it, print & log state.
         */
        else if( ( cur -> flags & IFF_PROMISC ) && ( dFlag == TRUE ) )
        {
            printf( "%6s: PROMISC [*]\n", cur -> data );

            if( ( sockfd = socket( AF_INET, SOCK_DGRAM, 0 ) ) == -1 )
            {
                perror( "socket" );
                return (-1);
             }

            strncpy( ifr.ifr_name, cur -> data, sizeof( ifr.ifr_name ) );

            if( ioctl( sockfd, SIOCGIFFLAGS, &ifr ) < 0 )
            {
                perror( "SIOCGIFFLAGS" );
                return (-1); 
             }

            ifr.ifr_flags &= ~(IFF_UP|IFF_RUNNING);

            if( ioctl( sockfd, SIOCSIFFLAGS, &ifr ) < 0 )
            {
                perror( "SIOCSIFFLAGS" );
                return (-1); 
             }

            if( ( writeLog( LOGWARNING, cur -> data, PROMISC_DISABLE ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        /*
         * Interface is up - print & log state.
         */
        else if( cur -> flags & IFF_UP )
        {
            printf( "%6s: normal\n", cur -> data );

            if( ( writeLog( LOGINFO, cur -> data, NORMAL ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        /*
         * Interface is down - print & log state.
         */
        else
        {
            printf( "%6s: *down*\n", cur -> data );

            if( ( writeLog( LOGINFO, cur -> data, DOWN ) ) != 0 )
            {
                fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
                return (-1);
             }
         }

        cur = cur -> next;
        i++;
     } 

    /*
     * Clean-up via close().
     */
    if( ( close( sockfd ) ) != 0 )
    {
        perror( "close" );
        return (-1); 
     }

    return (0);
}


/********************************************************
 * ifMetric()  - Generates and prints interface traffic *
 *               metrics for all configured interfaces. *
 *                                                      *
 * Parameters:   *ifList: first node of linked list of  *
 *               interface name/unit number pairings    *
 *               and corresponding flags.               *
 *                                                      *
 * Returns:      0 on success.                          *
 *               -1 on failure.                         *
 ********************************************************/

int ifMetric( struct ifList *list )
{
    struct stat fileAttrs;			/* /proc/net/dev file attributes. */ 
    char *filePath = NULL;			/* Path to /proc/net/dev file. */
    FILE *file = NULL;				/* File pointer for fopen(). */
    int status = 0;				/* /proc/net/dev file close() status. */
    char buf[1024] = "";			/* Character buffer. */
    struct ifList *cur = NULL;			/* Current node in linked list. */
    int conv = 0;				/* sscanf() string match count. */
    long long rxPackets = 0;			/* Received packet count. */
    long long txPackets = 0;			/* Transmitted packet count. */
    char ifMetrics[] = "interface metrics";	/* For metrics dump logging. */
    char *delim = NULL;				/* Ptr to matched character in string. */
    unsigned int ifIndex = 0;			/* Interface index. */

    /*
     * Exit if passed in interface list is empty. 
     */
    cur = list;

    if( cur -> data == NULL )
    {
        fprintf( stderr, "ifchk: ERROR: interface list is empty\n" );
        return (-1);
     }

    /*
     * Get /proc/net/dev attributes.
     */
    filePath = "/proc/net/dev";

    if( ( stat( filePath, &fileAttrs ) ) != 0 )
    {
        perror( "stat" );
        return (-1);
     }

    /*
     * Is /proc/net/dev zero bytes in length?
     */
    if( ( ( fileAttrs.st_size ) || ( fileAttrs.st_blocks ) ) != 0 )
    {
        fprintf( stderr, "ifchk: ERROR: /proc/net/dev file size is greater than 0\n" );
        return (-1);
     }

    /*
     * Is /proc/net/dev chowned UID/GID 0?
     */
    if( ( ( fileAttrs.st_uid ) || ( fileAttrs.st_gid ) ) != 0 )
    {
        fprintf( stderr, "ifchk: ERROR: /proc/net/dev is not owned by UID 0, GID 0\n" );
        return (-1);
     }

    /*
     * Open /proc/net/dev for reading.
     */
    if( ( file = fopen( "/proc/net/dev", "r" ) ) == NULL )
    {
        perror( "fopen" );
        return (-1);
     }

    memset( buf, '\000', sizeof( buf ) );

    /*
     * Read past 1st and 2nd line of /proc/net/dev header.
     */
    if( ! fgets(buf, sizeof( buf ), file) )
    {
        perror( "fgets" );
        return (-1);
     }

    if( ! fgets(buf, sizeof( buf ), file) )
    {
        perror( "fgets" );
        return (-1);
     }

    /*
     * Check /proc/net/dev format.
     */
    if( ( strstr( buf, "compressed" ) ) == NULL )
    {
        fprintf( stderr, "ifchk: ERROR: /proc/net/dev header format is not supported\n" );
        return (-1);
     }

    /*
     * Prepare to display metrics data.
     */
    printf( "***   Network Interface Metrics   ***\n" );
    printf( "Name    Index   RX-OK           TX-OK\n" ); 

    /*
     * Loop through /proc/net/dev metrics & output
     * one line of metrics data for each interface.
     */
    while( fgets( buf, sizeof( buf ), file ) )
    {
        /*
         * Is this the current interface?
         */
        if( ( strstr( buf, cur -> data ) ) != NULL )
        {
            delim = strchr( buf, ':' );

            /*
             * Select & process the correct output format string.
             */
            if( *( delim + 1 ) == ' ' )
            {
                conv = sscanf( buf,
                "%*s %*Lu %Lu %*lu %*lu %*lu %*lu %*lu %*lu %*Lu %Lu %*lu %*lu %*lu %*lu %*lu %*lu",
                 &rxPackets, &txPackets );
             }

            else
            {
                conv = sscanf( buf,
                "%*s %Lu %*lu %*lu %*lu %*lu %*lu %*lu %*Lu %Lu %*lu %*lu %*lu %*lu %*lu %*lu",
                 &rxPackets, &txPackets );
             }
         }

        else
        {
            fprintf( stderr, "ifchk: ERROR: current metrics do not describe current interface %s\n",
                                                                                         cur -> data );
            return (-1);
         }

        /*
         * Exit if we didn't match two strings (RX/TX packet counts).
         */
        if( conv != 2 )
        {
            fprintf( stderr, "ifchk: ERROR: /proc/net/dev parse error\n" );
            return (-1);
         }

        if( ( ifIndex = if_nametoindex( cur -> data ) ) == 0 )
        {
            perror( "if_nametoindex" );
            return (-1);
         }

        printf( "%-7s %-7d %-13Lu   %-13Lu\n", cur -> data, ifIndex, rxPackets, txPackets );
        
        conv = 0;

        if( cur -> next != NULL )
        {
            cur = cur -> next;
         }
     }

    /*
     * Clean-up via close().
     */
    if( ( status = fclose( file ) != 0 ) )
    {
        perror( "fclose" );
        return (-1);
     }

    /*
     * Log that a metrics dump was performed.
     */
    if( ( writeLog( LOGINFO, ifMetrics, NULLSTATE ) ) != 0 )
    {
        fprintf( stderr, "ifchk: ERROR: could not pass logging message to syslogd\n" );
        return (-1);
     }

    return (0);
}

