/************************************************************************
 * log.c                                                                *
 *                                                                      *
 * Copyright (C) 2002 - 2004 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/*************************************************************
 * Project ifchk -- Host based network interface promiscuous *
 *                  mode detection and handling.             *
 *                                                           *
 * Version:         0.95b2                                   *
 *                                                           *
 * --------------------------------------------------------- *
 *                                                           *
 * log.c:           ifchk logging subsystem.                 *
 *                                                           *
 * Module                                                    *
 * Functions:     - Performs logging of interface status and *
 *                  metrics dump invocation.                 *
 *                - Performs logging of privileged operation * 
 *                  (ie. interface shutdown) violations.     *
 *************************************************************/

/**********************
 * include directives *
 **********************/

#include "log.h"


/*****************************************************
 * writeLog()  - Performs logging of ifchk activity. *
 *                                                   *
 * Parameters:   priority: priority at which to log. *
 *               *object:  an interface name.        *
 *               ifState:  an interface state.       *
 *                                                   *
 * Returns:      0 on success.                       *
 *               -1 on failure.                      *
 *****************************************************/

int writeLog( int priority, char *object, int ifState )
{
    openlog( "ifchk", LOG_PID, LOG_USER );

    /*
     * Interface is down.
     */
    if( ( priority == LOGINFO ) && ( ifState == DOWN ) )
    {
        syslog( LOG_INFO, "%4s: *down*", object );
     }

    /*
     * Interface is normal/up.
     */
    else if( ( priority == LOGINFO ) && ( ifState == NORMAL ) )
    {
        syslog( LOG_INFO, "%4s: normal", object );
     }

    /*
     * Interface is promiscuous.
     */
    else if( ( priority == LOGWARNING ) && ( ifState == PROMISC ) )
    {
        syslog( LOG_WARNING, "%4s: PROMISC", object );
     }
    
    /*
     * Interface is promiscuous, disabled.
     */
    else if( ( priority == LOGWARNING ) && ( ifState == PROMISC_DISABLE ) )
    {
        syslog( LOG_WARNING, "%4s: PROMISC [*]", object );
     }
    
    /*
     * Interface status dump performed.
     */
    else if( ( priority == LOGINFO ) && ( ifState == STATSTATE ) )
    {
        syslog( LOG_INFO, "interface status: dump to stdout initiated by user %s (UID: %d)",
                                                                 pw -> pw_name, pw -> pw_uid );
     }

    /*
     * Metrics dump performed.
     */
    else if( ( priority == LOGINFO ) && ( ifState == NULLSTATE ) )
    {
        syslog( LOG_INFO, "interface metrics: dump to stdout initiated by user %s (UID: %d)",
                                                                  pw -> pw_name, pw -> pw_uid );
     }

    /*
     * Non privileged interface shutdown attempt.
     */
    else if( ( priority == LOGWARNING ) && ( ifState == NULLSTATE ) )
    {
        syslog( LOG_WARNING, "interface shutdown: attempt by non privileged user %s (UID: %d)",
                                                                    pw -> pw_name, pw -> pw_uid );
     }

    /*
     * Default: signals an error condition.
     */
    else
    {
        syslog( LOG_WARNING, "%4s: interface state logging failure", object );
        return(-1);
     }

    closelog();

    return (0);
}

