/************************************************************************
 * log.c                                                                *
 *                                                                      *
 * Copyright (C) 2002 - 2015 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/*************************************************************
 * Project ifchk -- Host based network interface promiscuous *
 *                  mode detection and handling.             *
 *                                                           *
 * Version:         1.0.1                                    *
 *                                                           *
 * --------------------------------------------------------- *
 *                                                           *
 * log.c:           ifchk logging subsystem.                 *
 *                                                           *
 * Functions:     - Routes ifchk log messages to syslog.     *
 *************************************************************/

/**********************
 * include directives *
 **********************/

#include "log.h"


/****************************************************
 * writeLog()  - Performs logging of ifchk activity *
 *               via syslog(3) and variable length  *
 *               argument lists.                    *
 *                                                  *
 * Parameters:   level: level at which to log.      *
 *               *fmt:  message to log.             *
 *                                                  *
 * Returns:      0 on success.                      *
 *               -1 on failure.                     *
 ****************************************************/

int writeLog( int level, const char *fmt, ... )
{
    va_list varArg;			/* For va_start(). */
    char argBuf[ARGBUFSZ] = "";		/* Variable argument list buffer. */
    char *nlPtr = NULL;			/* Ptr for trailing newline processing. */

    /*
     * Initialize the syslog(3) subsystem.
     * Write log messages to /dev/console via the LOG_CONS logging option if
     * we cannot write to syslogd(1M).
     * LOG_CONS is safe to use in daemon processes that have no controlling
     * terminal (i.e., ifchk running as a daemon), since syslog(3) forks
     * before opening the console.
     */
    openlog( "ifchk", LOG_CONS | LOG_PID, LOG_USER );

    /*
     * We are expecting log messages
     * at these log levels only.
     */
    if( ( level == LOG_INFO ) ||
        ( level == LOG_NOTICE ) ||
        ( level == LOG_WARNING ) ||
        ( level == LOG_CRIT ) )
    {
        memset( argBuf, '\000', sizeof( argBuf ) );

	/*
	 * Initialize the variable length
	 * argument list facility.
	 */
        va_start( varArg, fmt );

	/*
	 * Prepare to send our log
	 * message to syslog(3).
	 */
        if( vsnprintf( argBuf, sizeof( argBuf ), fmt, varArg ) == -1 )
        {
            errHandler( "ERROR: vsnprintf(): could not write to character buffer: %s\n",
                                                                       strerror( errno ) );
            return (-1);
         }

        argBuf[ARGBUFSZ - 1] = '\000';

	/*
	 * Did the log message overrun
	 * our logging buffer?
	 */
        if( ( nlPtr = strchr( argBuf, '\n' ) ) == NULL )
        {
            errHandler( "%s", "ERROR: could not log to syslogd: ifchk log buffer full\n" );
            return (-1);
         }

	/*
	 * If not, strip the trailing newline
	 * and write the log message to syslog.
	 */
        else
        {
            *nlPtr = '\000';

            syslog( level, argBuf );
         }
     }

    else
    {
        syslog( LOG_WARNING, "ERROR: unrecognized syslog level \"%d\"", level );
        return (-1);
     }

    /*
     * Deallocate list memory after use.
     */
    va_end( varArg );

    /*
     * Close the syslog(3) subsystem.
     */
    closelog();

    return (0);
}
