/************************************************************************
 * evtloop.c                                                            *
 *                                                                      *
 * Copyright (C) 2002 - 2020 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/*************************************************************
 * Project ifchk -- Host based network interface promiscuous *
 *                  mode detection and handling.             *
 *                                                           *
 * Version:         1.1.2                                    *
 *                                                           *
 * --------------------------------------------------------- *
 *                                                           *
 * evtloop.c:       ifchk program event loop.                *
 *                                                           *
 * Module                                                    *
 * Functions:     - Waits for a predefined signal, and       *
 *                  acts accordingly upon receipt.           *
 *************************************************************/

/**********************
 * include directives *
 **********************/

#include "evtloop.h"


/*******************************************************
 * evtLoop() - Main program event loop.                *
 *                                                     *
 * Parameters: mask:    signal mask.                   *
 *             promisc: promiscuous interface shutdown *
 *                      switch (enabled/disabled.)     *
 *                                                     *
 * Returns:                                            *
 *******************************************************/

int evtLoop( sigset_t mask, int promisc )
{
    static sigset_t sigMask;    /* For POSIX signal set. */
    static int sigNum;          /* Signal number. */
    int disablePromisc = 0;     /* Enable/disable promiscuous interface shutdown. */

    sigMask = mask;

    disablePromisc = promisc;

    /*
     * Program signal event loop.
     */
    for( ;; )
    {
        /*
         * We sigwait() here for signals. This loop determines our res-
         * ponse to the reception of SIGINT, SIGTERM, SIGUSR1 & SIGUSR2.
         */
        if( sigwait( &sigMask, &sigNum ) != 0 )
        {
            errHandler( "ERROR: sigwait(): signal selection failed: %s\n", strerror( errno ) );
            exit( EXIT_FAILURE );
         }

        if( sigNum == SIGHUP )
        {
            hFlag = TRUE; 
         }

        /*
         * We caught SIGINT or SIGTERM - exit, and return to main() context.
         */
        if( ( sigNum == SIGINT ) || ( sigNum == SIGTERM ) )
        {
            break;
         }

        /*
         * We caught SIGUSR1 - perform an interface
         * status scan when we call ifStat(), below.
         */
        if( sigNum == SIGUSR1 )
        {
            iFlag = FALSE;

            /*
             * If disablePromisc is set, report and shutdown promiscuous interfaces.
             */
            if( disablePromisc == 1 )
            {
                dFlag = TRUE;
             }

            /*
             * If disablePromisc is not set, report promiscuous interfaces only.
             */
            else
            {
                nullFlag = TRUE;
             }
         }

        /*
         * We caught SIGUSR2 - perform an interface
         * metrics dump when we call ifStat(), below.
         */
        if( sigNum == SIGUSR2 )
        {
            dFlag = FALSE;
            nullFlag = FALSE;
            iFlag = TRUE;
         }

        /*
         * Call ifStat().
         */
        if( ifStat() != 0 )
        {
            exit( EXIT_FAILURE );
         }
     }

    return (0);
}
