/************************************************************************
 * bondutils.c                                                          *
 *                                                                      *
 * Copyright (C) 2002 - 2024 Joshua Birnbaum <engineer@noorg.org>.      *
 * All Rights Reserved.                                                 *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published    *
 * by the Free Software Foundation; either version 2 of the License, or *
 * (at your option) any later version.                                  *
 *                                                                      *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS    *
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY      *
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; see the file COPYING. If not, write to the  *
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, *
 * MA 02111-1307, USA.                                                  *
 ************************************************************************/

/********************************************************************
 * Project ifchk -- Host based network interface promiscuous        *
 *                  mode detection and handling.                    *
 *                                                                  *
 * Version:         1.1.4                                           *
 *                                                                  *
 * ---------------------------------------------------------------- *
 *                                                                  *
 * bondutils.c:     Logical bonding interface discovery.            *
 *                                                                  *
 * Module                                                           *
 * Functions:     - Gets bonding interface slave names.             *
 *                - Gets currently active physical slave interface. *
 ********************************************************************/

/**********************
 * include directives *
 **********************/

#include "bondutils.h"


/********************************************************************
 * getSlaves() - Build a list comprising all bonding interface      *
 *               slaves and currently active slave interface        *
 *                                                                  *
 * Parameters:   *path: path to logical bonding interface flags.    *
 *               *name: logical bonding interface name/unit number. *
 *               *file: full path to slave interface sysfs file.    *
 *                                                                  *
 * Returns:      List of Bonding interface slaves + active slave.   *
 *               -1 on failure.                                     *
 ********************************************************************/

char * getSlaves( char *path, char *name, char *file )
{
    FILE *bondslavefd = NULL;                   /* File pointer to bonding interface slaveIf. */
    char slaveFile[BONDBUFSZ] = "";             /* Pathname of sysfs filesystem bonding interface slaves. */
    char slaveActPath[BONDBUFSZ];               /* Pathname of sysfs active slave interface. */
    char slaveIf[BONDBUFSZ] = "";               /* Bonding interface slaves. */
    char slaveAct[BONDBUFSZ] = "";              /* Buffer for sysfs active slave interface. */
    static char slaveInfo[SLAVEBUFSZ] = "";     /* Bonding interface slaves + active slave. */
    char *nlPtr = NULL;                         /* Ptr for trailing newline processing. */
    struct fsObj slaveObject;                   /* for statFile(). */

    /*
     * Reset all statFile() file checks.
     */
    slaveObject.openRead = FALSE;
    slaveObject.procLen = FALSE;
    slaveObject.procOwner = FALSE;
    slaveObject.procMode = FALSE;
    slaveObject.regLen = FALSE;
    slaveObject.regMode = FALSE;

    /*
     * Preset the file query policy for fileStat(), below. Specifically,
     * we want to probe the sysfs file attributes of bonding interface
     */
    slaveObject.openRead = TRUE;
    slaveObject.regLen = TRUE;
    slaveObject.regMode = TRUE;

    /*
     * Prepare to securely open the slave sysfs file.
     */
    if( ( bondslavefd = statFile( file, slaveObject ) ) == NULL )
    {
        return (NULL);
     }

    /*
     * Once opened, read the file's contents (e.g., "em1 em2").
     */
    while( ( fgets( slaveIf, sizeof( slaveIf ), bondslavefd ) ) != NULL )
    {
        if( ( nlPtr = strchr( slaveIf, '\n' ) ) )
        {
            *nlPtr = '\000';
         }
     }

    /*
     * Once read, close the slave sysfs file.
     */
    if( fclose( bondslavefd ) != 0 )
    {
        errHandler( "ERROR: fclose(): could not close %s: %s\n", slaveFile, strerror( errno ) );
        return (NULL);
     }

    /*
     * Build a full path to the active slave interface sysfs file.
     * Manually null terminate slaveActPath[] for redundancy.
     */
    memset( slaveActPath, '\000', sizeof( slaveActPath ) );
    strncpy( slaveActPath, "/sys/devices/virtual/net/", sizeof( slaveActPath ) );
    slaveActPath[BONDBUFSZ - 1] = '\000';
    strncat( slaveActPath, name, sizeof( slaveActPath ) - strlen( slaveActPath ) - 1 );
    strncat( slaveActPath, "/bonding/active_slave", sizeof( slaveActPath ) - strlen( slaveActPath ) -1 );

    /*
     * Prepare to securely open the active slave sysfs file.
     */
    if( ( bondslavefd = statFile( slaveActPath, slaveObject ) ) == NULL )
    {
       return (NULL);
     }

    /*
     * Once opened, read the file's contents (e.g., "em1").
     */
    while( ( fgets( slaveAct, sizeof( slaveAct ), bondslavefd ) ) != NULL )
    {
        if( ( nlPtr = strchr( slaveAct, '\n' ) ) )
        {
            *nlPtr = '\000';
         }
     }

    /*
     * Once read, close the slave sysfs file.
     */
    if( fclose( bondslavefd ) != 0 )
    {
        errHandler( "ERROR: fclose(): could not close %s: %s\n", slaveFile, strerror( errno ) );
        return (NULL);
     }

    memset( slaveInfo, '\000', sizeof( slaveInfo ) );
    strncpy( slaveInfo, slaveIf, sizeof( slaveInfo) );
    slaveInfo[SLAVEBUFSZ - 1] = '\000';
    strncat( slaveInfo, " ", sizeof( slaveInfo) - strlen( slaveInfo ) - 1 );
    strncat( slaveInfo, slaveAct, sizeof( slaveInfo ) - strlen( slaveInfo ) - 1 );

    return( slaveInfo );
}
